*! version 1.0.0  1aug2014  dcs

* based on official Stata's -var_p- and -svar_p-

program define svarih_bfa_p100, sortpreserve

version 11.2

syntax newvarlist [if] [in] ,       ///  newvarlist is used as a stub when option hdecomp is used
                                    ///  e.g. if 'hdecomp hdshock(eq1)', then the HD for the shock in eq1 of all endogvars is calced,
                                    ///  the new varnames being 'stub'eqname
                                    ///  must parse using 'newvarname' as this automatically defines `typlist'
            [ xb                    ///  default
              Residuals             ///
              Shocks                ///  default
              HDecomp               ///
              HDShock(string)   ///
              HDBeg(string)         /// may be date or date number
              HDEnd(string)         /// may be date or date number; TODO: also allow to specify # of periods
              EQuation(string)      ///
              ]

    qui tsset
    local tvar   `r(timevar)'  // locals needed for option -hdecomp-
    local unit1  `r(unit1)'
    local tsfmt  `r(tsfmt)'

    marksample touse, novarlist

    if "`e(cmd)'" != "svarih" | "`e(method)'"!="BFanelli" {
        di as err "svarih_bfa_p100 only works after -svarih bfanelli-"
        exit 301
    }

    local nstats : word count `xb' `residuals' `shocks' `hdecomp'
    if `nstats' > 1 {
        di as err "more than one statistic specified"
        exit 198
    }

    if "`xb'`residuals'`shocks'`equation'"!="" & `"`hdecomp'`hdbeg'`hdend'`hdshock'"'!="" {
        di as err `"Invalid combination of options."'
        exit 198
    }

    if "`xb'`residuals'`shocks'`hdecomp'" == "" {
        local xb  xb
        di as txt "(option xb assumed; fitted values)"
    }

    if "`equation'" != "" {
        if `: word count `equation''>1 {
            disp as error `"Option 'equation' may only contain one token."'
            exit 198
        }
    }
    else {
        local equation "#1"
    }
    
    Depname depname : `equation'
    if "`depname'" == "" {
        di as error "`equation' is not a valid equation name"
        exit 198
    }   

    local rgmvar     `e(rgmvar)'
    local regimes    `e(regimes)'
    local numregimes `e(numregimes)'
    forvalues s=1/`numregimes' {
        tempname b`s'
        matrix `b`s'' = e(b_var`s')
    }

    capture confirm variable `rgmvar'
    if _rc {
        disp as error `"regime variable `rgmvar' not found"'
        exit 111
    }
    markout `touse' `rgmvar'
    // -svarih bfa- requires rgmvar for resids, shocks, hdecomp

    qui levelsof `rgmvar' if `touse' , local(regimes_touse)   // rgmvar of bfa and llu may not contain values in `touse' other than e(regimes)
    if !`: list regimes_touse in regimes' {
        disp as error `"Observations selected contain values for regime variable `rgmvar'"'
        disp as error `"that do not occur in the estimation sample."'
        exit 459
    }
    
    if "`xb'`residuals'" != "" {
        tempvar xbtemp xbpart
        qui gen double `xbtemp' = .
        forvalues s=1/`numregimes' {
            capture drop `xbpart'
            qui matrix score double `xbpart' = `b`s'' if `touse' & `rgmvar'==`s', equation(`depname')
            qui replace `xbtemp' = `xbpart' if `touse' & `rgmvar'==`s'
        }
        if "`xb'"!="" {
            gen `typlist' `varlist' = `xbtemp' if `touse'
            label variable `varlist' "svarih bfa: fitted values, equation '`depname''"
        }
        else {
            gen `typlist' `varlist' = `depname' - `xbtemp' if `touse'
            label variable `varlist' "svarih bfa: residual, equation '`depname''"
        }
        exit
    }

    tempname B E2 E3 E4
    matrix `B' = e(B)
    forvalues s=2/`numregimes' {
        matrix `E`s'' = e(E`s')
    }

    if "`shocks'" != "" {  // calc shock series: B^(-1) * u_t,s = e_t,s  ;  (B+E_s)^(-1) * u_t,s = e_t,s 

        tempname BEinv rowvec
        
        local numendog = rowsof(`B')
        forvalues i=1/`numendog' {   // need all residuals to compute one shock
            tempname u`i'
            qui svarih_bfa_p100 double `u`i'' if `touse', residuals eq(#`i')
            local cnames `cnames' `u`i''
        }

        tempvar tempshock
        qui gen `typlist' `varlist' = .
        forvalues s=1/`numregimes' {
            if `s'==1 {
                matrix `BEinv' = inv(`B')
            }
            else {
                matrix `BEinv' = inv(`B'+`E`s'')
            }
            matrix `rowvec' = `BEinv'["`depname'",1...]
            matrix colnames `rowvec' = `cnames'

            capture drop `tempshock'
            qui matrix score `typlist' `tempshock' = `rowvec' if `touse' & `rgmvar'==`s'
            qui replace `varlist' = `tempshock' if `touse' & `rgmvar'==`s'
        }
        label variable `varlist' "svarih bfa: estimated shock, equation '`depname''"
    }

    if "`hdecomp'"!="" {

        local stub `varlist'

        local depvar `e(depvar)'
        local exog   `e(exog)'
        local mlag   `e(mlag)'
        
        if "`hdbeg'"=="" | "`hdend'"=="" {
            disp as error `"Option 'hdecomp' must be accompanied by options 'hdbeg' and 'hdend'."'
            exit 198
        }
        
        if !inlist("`unit1'","y","h","q","m","d","b") {        // TODO: make it work for intraday-freqs
            disp as error `"calculation of historical decompositions"
            disp as error `"are implemented only for frequencies y, h, q, m, d, b."'  // some calcs below require integer values of `tvar'
            exit 459
        }
        
        foreach curopt in hdbeg hdend {
            capture confirm number ``curopt''
            if _rc {
                if "`unit1'"=="b" {
                    capture local num`curopt' = bofd(substr("`tsfmt'",4,.),td(``curopt''))
                }
                else {
                    capture local num`curopt' = t`unit1'(``curopt'')
                }
                if _rc {
                    disp as error `"Could not process date argument to option '`curopt''."'
                    exit 198
                }
            }
            else {
                local num`curopt' ``curopt''
            }
            
            qui count if `tvar'==`num`curopt''
            if `r(N)'!=1 {
                disp as error `"Time value `num`curopt'' from option '`curopt'' does not or not uniquely identify an observation."'
                exit 459
            }
        }        

        if `numhdbeg'>`numhdend' {
            disp as error `"Beginning date for historical decomposition after ending date."'
            exit 198
        }

        // check that periods that are necessary to compute the step 1 forecast have all nonmissings
        //   also check the exog vars which must have non-missings over the entire prediction period, otherwise missings will be predicted from the occurrence of an exogvar missing on
        tempvar rm rm2
        qui egen int `rm' = rowmiss(`depvar') if inrange(`tvar', `=`numhdbeg'-`mlag'', `=`numhdbeg'-1')
        capture assert `rm' == 0              if inrange(`tvar', `=`numhdbeg'-`mlag'', `=`numhdbeg'-1')
        if _rc {
            disp as error `"Lags of endogenous variables necessary to compute the step 1 forecast contain missing values."'
            exit 416
        }
        tsrevar `exog' , substitute
        local exogtmp `r(varlist)'
        qui egen `rm2' = rowmiss(`exogtmp') if inrange(`tvar',`numhdbeg',`numhdend')
        capture assert `rm2' == 0           if inrange(`tvar',`numhdbeg',`numhdend')
        if _rc {
            disp as error `"Exogenous variables contain missings in forecast span."'
            exit 416
        }
        
        if "`hdshock'"!="" {  // if empty, generate baseline forecast

            qui su `tvar' if e(sample), meanonly
            local smplbeg `r(min)'
            local smplend `r(max)'
            if `numhdbeg'<`smplbeg' | `numhdend'>`smplend' {
                disp as error `"Arguments to options 'hdbeg' and/or 'hdend' not in sample range"'
                exit 198
            }

            if `: word count `hdshock''>1 {
                disp as error `"Option 'hdshock' may only contain one token."'
                exit 198
            }
            Depname shockeq : `hdshock'
            if "`shockeq'" == "" {
                di as error "`hdshock' is not a valid equation name."
                exit 198
            }   

            // check for non-existence of output variables (the forecast variables) and proper naming of variables
            foreach curendog of local depvar {
                capture confirm name `stub'`curendog'
                if _rc {
                    disp as error `"stub `stub' invalid: `stub'`curendog' is not a proper variable name."'
                    exit 198
                }
                capture confirm new variable `stub'`curendog'
                if _rc {
                    disp as error `"Variable `stub'`curendog' already defined."'
                    exit 110
                }
            }

            tempname shock_i
            qui predict `shock_i', eq(`hdshock')

            // gen components (i.e. a vector) of red-form resid u, `rp_`curendog''
            //     where "rp": residual part
            //     gen neqs respart variables, each calc as elem (eqnum,i) of B (or B+E`s') multiplied by `shock_i'
            //     hence I only need col i of B (or B+E`s'), however, this element changes for each regime s so I have to loop over it
            tempname BE jnk
            foreach curendog of local depvar {
                tempvar  rp_`curendog'
                qui gen double `rp_`curendog'' = .
            }
            forvalues s=1/`numregimes' {
                if `s'==1 {
                    matrix `BE' = `B'
                }
                else {
                    matrix `BE' = `B'+`E`s''
                }
                foreach curendog of local depvar {
                    matrix `jnk' = `BE'["`curendog'","`shockeq'"]
                    scalar shock_mlt = `jnk'[1,1]
                    qui replace `rp_`curendog'' = `shock_i' * shock_mlt if `touse' & `rgmvar'==`s'
                }
            }
        }
        
        foreach curendog of local depvar {
            qui gen double `stub'`curendog' = .
            qui replace `stub'`curendog' = `curendog' if inrange(`tvar', `=`numhdbeg'-`mlag'', `=`numhdbeg'-1')  // note that tvar units must be integers!
        }
        
        preserve
        tempname regimeend
        tempvar pvar
        qui gen byte `pvar'=1
        qui tsset `pvar' `tvar'
        topfilter , catvar(`rgmvar')
        qui drop if t_end < `numhdbeg'
        qui drop if t_beg > `numhdend'
        qui replace t_end = `numhdend' in l
        mkmat t_end rgmvar , matrix(`regimeend') // matrix has end of regime in col 1, regime encoding in col 2
                                                 // TODO/POTENTIAL BUG: think what happens if rgmvar has gaps
        restore

        local bnames : colfullnames `b1'  // colnames are the same for all sub-VARs

        foreach curendog of local depvar {
            local bnames : subinstr local bnames ".`curendog' " ".`stub'`curendog' ", all  // note the dot and the space
        }

        local numhds = rowsof(`regimeend')       // # of switches of regimes within HD forecast period

        forvalues hds=1/`numhds' {
            if `hds'==1 {
                local numhdbeg_tmp `numhdbeg'
            }
            else {
                local numhdbeg_tmp `=`numhdend_tmp'+1'
            }
            local numhdend_tmp = `regimeend'[`hds',1]  // first col contains t_end
            
            // replace column names in b`s': names of endog vars are now the names of simulated vars
            local s = `regimeend'[`hds',2]  // second col contains regime encoding
            matrix colnames `b`s'' = `bnames'

            // snippet copied & modified from offical Stata's _varsim.ado
            local j 1
            foreach curendog of local depvar {
                local curendog_new `stub'`curendog'
                local eqj : word `j' of `depvar'
                local scr`j' "score  `curendog_new' = `b`s'', eq(#`j') "
                if "`hdshock'"!="" local upd`j' "update `curendog_new' = `curendog_new' + `rp_`curendog''"  // this line distinguishes baseline from shock_i forecasts
                local j = `j' + 1
            }

            _byobs {
                `scr1'
                `upd1'
                `scr2'
                `upd2'
                `scr3'
                `upd3'
                `scr4'
                `upd4'
                `scr5'
                `upd5'
                `scr6'
                `upd6'
                `scr7'
                `upd7'
                `scr8'
                `upd8'
                `scr9'
                `upd9'
                `scr10'
                `upd10'
                `scr11'
                `upd11'
                `scr12'
                `upd12'
                `scr13'
                `upd13'
                `scr14'
                `upd14'
                `scr15'
                `upd15'
                `scr16'
                `upd16'
                `scr17'
                `upd17'
                `scr18'
                `upd18'
                `scr19'
                `upd19'
                `scr20'
                `upd20'
            } if inrange(`tvar',`numhdbeg_tmp',`numhdend_tmp')
        }

        foreach curendog of local depvar {  // final time series start at hdbeg-1, as in -fcast compute-
            qui replace `stub'`curendog' = . if `tvar'<`=`numhdbeg'-1'
        }
    }

end

*** --------------------------------- SUBROUTINES -----------------------------------------

// modified official Stata routine: returns equation (=variable) name 
program define Depname

    args    depname  /// macro to hold dependent variable name
            colon    /// ":"
            eqopt    //  equation name or #number

    if substr("`eqopt'",1,1) == "#" {
        local eqnum =  substr("`eqopt'", 2,.)
        local dep : word `eqnum' of `e(depvar)'
        c_local `depname' `dep'
        exit
    }
        
    local eqlist  "`e(depvar)'"  // distinction b/w eqlist and deplist only matters when depvar has ts ops, which -svar- allows for ; -svarih- does not
    local deplist "`e(depvar)'"
    local i 1
    while "`dept'" == "" & "`eqlist'" != "" {
        gettoken eqn eqlist : eqlist
        if "`eqn'" == "`eqopt'" {
            local dept : word `i' of `deplist'
            c_local `depname' `dept'
        }
        local i = `i' + 1
    }

end





program define topfilter, nclass

version 10.1
syntax  [, Catvar(varlist numeric min=1 max=1)  ///
           BEGOpen                              ///
           ENDOpen]

preserve  // restore data set if error occurs during execution of topfilter
          // also necessary b/c -tsset- settings are changed in the code

qui tsset
local pvar    `r(panelvar)'
local tvar    `r(timevar)'
local freq    `r(unit1)'
local tsfmt   `r(tsfmt)'
local tdeltas `r(tdeltas)'

local calname ""
if "`freq'"=="b" local calname = substr("`tsfmt'",4,.)

if "`tvar'" == "" | "`pvar'" == "" {
    disp as error `"Panel variable or time variable not set."'
    exit 111
}

if !inlist("`freq'","c","C","d","b","w","m","q","h","y") {
    disp as error "tsset's r(unit1) must be set to either 'c', 'C', 'd', 'b', 'w', 'm', 'q', 'h', or 'y'."
    exit 451
}

local pvarvarl:  variable label `pvar'
local pvarvll:   value label `pvar'

if "`catvar'"!="" {
    if inlist("`catvar'","t_beg","t_end") {
        tempname catvar2
        ren `catvar' `catvar2'
        local catvar `catvar2'
    }
    qui drop if `catvar' >= .
    if `c(N)'==0 exit
    local catvarvll: value label `catvar'
    local catvarvarl: variable label `catvar'
}

capture confirm new variable t_end
if _rc {
    disp as error `"Data set may not contain a variable named 't_end'."'
    exit 9
}

tempvar tdiff group chgdummy

// the two commented out statements did not account for different deltas, e.g. delta(2 days)
// qui by `pvar': gen `tdiff' = `tvar'[_n]-`tvar'[_n-1]
// qui by `pvar': replace `tdiff'=1 if _n==1  // the previous statement leaves first obs of each crosec missing

qui by `pvar': gen `tdiff' = l.`tvar'
qui replace `tdiff'=1 if `tdiff'<.

if "`catvar'"!="" {
    qui gen byte `chgdummy' = ((`pvar'[_n]!=`pvar'[_n-1]) | (`tdiff'!=1) | (`catvar'[_n]!=`catvar'[_n-1]))
}
else {
    qui gen byte `chgdummy' = ((`pvar'[_n]!=`pvar'[_n-1]) | (`tdiff'!=1))
}
qui gen `group' = sum(`chgdummy')

qui tsset, clear

if "`tvar'"!="t_beg" ren `tvar' t_beg

qui clonevar t_end = t_beg

if "`catvar'"!="" local catvarpart (first) `catvar'
qui collapse (first) `pvar' (first) t_beg (last) t_end `catvarpart', by(`group')
qui drop `group'

sort `pvar' t_beg

if "`begopen'"!="" {
    qui by `pvar': replace t_beg=. if _n==1
}
if "`endopen'"!="" {
    qui by `pvar': replace t_end=. if _n==_N
}

format t_beg t_end %t`freq'`calname'
label variable `pvar'   "`pvarvarl'"
label variable t_beg    ""
label variable t_end    ""
if "`catvar'"!="" label variable `catvar' "`catvarvarl'"

// must re-apply value label after collapse
if "`pvarvll'"  !="" label values `pvar'   `pvarvll'
if "`catvarvll'"!="" label values `catvar' `catvarvll'

if "`pvar'"!="groupvar" ren `pvar' groupvar

chkpfilter, wide novarnamechk
restore, not

end



